<?php
/**
 * Plugin Name: Gudauri Map Section
 * Description: Renders the Gudauri road map section with editable texts and pins. Use the shortcode [gudauri_map].
 * Version: 1.0.0
 * Author: Codex
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Default texts for the map columns.
 */
function gudauri_map_default_settings() {
    return array(
        'tb_title' => 'Tbilisi Gudauri Road Map',
        'tb_meta'  => '2 HR 20 MIN (120 KM)',
        'tb_body'  => 'The road to Gudauri offers one of Georgia’s most scenic drives. Just 120 kilometers north of Tbilisi, the journey follows the historic Georgian Military Highway. In about an hour and a half, you travel from the lowlands into alpine landscapes, climbing through valleys and mountain passes and discovering landmarks along the way.',

        'mt_title' => 'Mtskheta',
        'mt_meta'  => '50 MIN (26 KM)',
        'mt_body'  => 'Mtskheta is Georgia’s ancient capital and a UNESCO site. Visit Svetitskhoveli Cathedral and Jvari Monastery perched above the Aragvi and Mtkvari rivers.',

        'zhi_title' => 'Zhinvali',
        'zhi_meta'  => '1 HR 10 MIN (55 KM)',
        'zhi_body'  => 'Zhinvali Reservoir paints a vivid turquoise ribbon along the road. Stop for photos from the roadside viewpoints over the dam and mountains.',

        'ana_title' => 'Ananuri',
        'ana_meta'  => '1 HR 20 MIN (66 KM)',
        'ana_body'  => 'The Ananuri Fortress overlooks the reservoir. Climb its towers to take in sweeping views and wander the stone churches inside the complex.',

        'pas_title' => 'Pasanauri',
        'pas_meta'  => '1 HR 40 MIN (89 KM)',
        'pas_body'  => 'A mountain town where the Aragvi rivers meet. Warm up with khinkali and follow the river bends deeper toward the Caucasus.',

        'map_url' => '',
        'pin_mt' => 'Mtskheta',
        'pin_zhi' => 'Zhinvali',
        'pin_ana' => 'Ananuri',
        'pin_pas' => 'Pasanauri',

        // Georgian (ka) fallbacks are empty; frontend will fall back to base if unset.
        'tb_title_ka' => '',
        'tb_meta_ka'  => '',
        'tb_body_ka'  => '',

        'mt_title_ka' => '',
        'mt_meta_ka'  => '',
        'mt_body_ka'  => '',

        'zhi_title_ka' => '',
        'zhi_meta_ka'  => '',
        'zhi_body_ka'  => '',

        'ana_title_ka' => '',
        'ana_meta_ka'  => '',
        'ana_body_ka'  => '',

        'pas_title_ka' => '',
        'pas_meta_ka'  => '',
        'pas_body_ka'  => '',

        'map_url_ka' => '',
        'pin_mt_ka' => '',
        'pin_zhi_ka' => '',
        'pin_ana_ka' => '',
        'pin_pas_ka' => '',
    );
}

/**
 * Get settings merged with defaults.
 */
function gudauri_map_get_settings() {
    $defaults = gudauri_map_default_settings();
    $saved    = get_option('gudauri_map_settings', array());
    return wp_parse_args($saved, $defaults);
}

/**
 * Enqueue assets for the map section.
 */
function gudauri_map_enqueue_assets() {
    $plugin_url = plugin_dir_url(__FILE__);

    wp_enqueue_style(
        'gudauri-map-fonts',
        'https://fonts.googleapis.com/css2?family=Instrument+Serif:ital,wght@0,400;1,400&family=Open+Sans:wght@400;600&display=swap',
        array(),
        null
    );

    wp_enqueue_style(
        'gudauri-map-style',
        $plugin_url . 'assets/css/gudauri-map.css',
        array('gudauri-map-fonts'),
        '1.0.0'
    );

    wp_enqueue_script(
        'gudauri-map-script',
        $plugin_url . 'assets/js/gudauri-map.js',
        array('jquery'),
        '1.0.0',
        true
    );

    wp_localize_script('gudauri-map-script', 'GudauriMapAssets', array(
        'backArrow' => $plugin_url . 'assets/back-arrow.svg',
    ));
}

function gudauri_map_admin_enqueue($hook) {
    if ($hook !== 'settings_page_gudauri-map') {
        return;
    }

    wp_enqueue_media();
    wp_register_script('gudauri-map-admin', '', array('jquery'), '1.0.0', true);
    wp_enqueue_script('gudauri-map-admin');

    $script = <<<JS
(function($) {
  $(document).on('click', '.gudauri-map-upload', function(e) {
    e.preventDefault();
    var targetId = $(this).data('target');
    if (!targetId) {
      return;
    }
    var frame = wp.media({
      title: 'Select map image',
      button: { text: 'Use this image' },
      library: { type: 'image' },
      multiple: false
    });
    frame.on('select', function() {
      var attachment = frame.state().get('selection').first();
      if (!attachment) {
        return;
      }
      var data = attachment.toJSON();
      if (data && data.url) {
        $('#' + targetId).val(data.url);
      }
    });
    frame.open();
  });
})(jQuery);
JS;

    wp_add_inline_script('gudauri-map-admin', $script);
}
add_action('admin_enqueue_scripts', 'gudauri_map_admin_enqueue');

/**
 * Shortcode renderer.
 */
function gudauri_map_shortcode() {
    gudauri_map_enqueue_assets();

    $settings   = gudauri_map_get_settings();
    $plugin_url = plugin_dir_url(__FILE__);
    $lang_slug  = function_exists('pll_current_language') ? pll_current_language('slug') : '';
    $lang_suffix = ($lang_slug === 'ka' || $lang_slug === 'ka_GE') ? '_ka' : '';

    $get_text = function ($key) use ($settings, $lang_suffix) {
        $localized_key = $key . $lang_suffix;
        if ($lang_suffix && !empty($settings[$localized_key])) {
            return $settings[$localized_key];
        }
        return $settings[$key] ?? '';
    };

    $map_url_setting = $get_text('map_url');
    $map_url = $map_url_setting ? esc_url($map_url_setting) : esc_url($plugin_url . 'assets/map.svg');

    ob_start();
    ?>
    <div class="gudauri-map" style="--gudauri-map-url: url('<?php echo $map_url; ?>');">
      <section class="section map-section">
        <div class="section-content relative">
          <div class="row row-full-width">

            <div class="col tbilisi">
              <div class="col-inner">
                <div class="text dineat">
                  <h2><?php echo esc_html($get_text('tb_title')); ?></h2>
                </div>
                <p class="meta"><?php echo esc_html($get_text('tb_meta')); ?></p>
                <p class="body"><?php echo wp_kses_post($get_text('tb_body')); ?></p>
              </div>
            </div>

            <div class="col mtskheta">
              <div class="col-inner">
                <div class="text">
                  <h2><?php echo esc_html($get_text('mt_title')); ?></h2>
                </div>
                <p class="meta"><?php echo esc_html($get_text('mt_meta')); ?></p>
                <p class="body"><?php echo wp_kses_post($get_text('mt_body')); ?></p>
              </div>
            </div>

            <div class="col zhinvali">
              <div class="col-inner">
                <div class="text">
                  <h2><?php echo esc_html($get_text('zhi_title')); ?></h2>
                </div>
                <p class="meta"><?php echo esc_html($get_text('zhi_meta')); ?></p>
                <p class="body"><?php echo wp_kses_post($get_text('zhi_body')); ?></p>
              </div>
            </div>

            <div class="col ananuri">
              <div class="col-inner">
                <div class="text">
                  <h2><?php echo esc_html($get_text('ana_title')); ?></h2>
                </div>
                <p class="meta"><?php echo esc_html($get_text('ana_meta')); ?></p>
                <p class="body"><?php echo wp_kses_post($get_text('ana_body')); ?></p>
              </div>
            </div>

            <div class="col pasanauri">
              <div class="col-inner">
                <div class="text">
                  <h2><?php echo esc_html($get_text('pas_title')); ?></h2>
                </div>
                <p class="meta"><?php echo esc_html($get_text('pas_meta')); ?></p>
                <p class="body"><?php echo wp_kses_post($get_text('pas_body')); ?></p>
              </div>
            </div>

            <div class="col mp">
              <div class="col-inner">
                <div class="banner maps">
                  <div class="banner-layers container">
                    <div class="map-frame">
                      <div class="map-overlay">
                        <div class="map-cross cross-mt" aria-hidden="true"></div>
                        <div class="map-cross cross-zhi" aria-hidden="true"></div>
                        <div class="map-cross cross-ana" aria-hidden="true"></div>
                        <div class="map-cross cross-pas" aria-hidden="true"></div>

                        <div class="pin pin-mt"><a class="button primary is-link lowercase pins mt-pin"><span><?php echo esc_html($get_text('pin_mt')); ?></span></a></div>
                        <div class="pin pin-zhi"><a class="button primary is-link lowercase pins zhi-pin"><span><?php echo esc_html($get_text('pin_zhi')); ?></span></a></div>
                        <div class="pin pin-ana"><a class="button primary is-link lowercase pins ana-pin"><span><?php echo esc_html($get_text('pin_ana')); ?></span></a></div>
                        <div class="pin pin-pas"><a class="button primary is-link lowercase pins pas-pin"><span><?php echo esc_html($get_text('pin_pas')); ?></span></a></div>
                      </div>
                      <div class="map-graphic" aria-hidden="true"></div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

          </div>
        </div>
      </section>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('gudauri_map', 'gudauri_map_shortcode');

/**
 * Admin settings page.
 */
function gudauri_map_admin_menu() {
    add_options_page(
        __('Gudauri Map', 'gudauri-map'),
        __('Gudauri Map', 'gudauri-map'),
        'manage_options',
        'gudauri-map',
        'gudauri_map_render_settings_page'
    );
}
add_action('admin_menu', 'gudauri_map_admin_menu');

function gudauri_map_render_settings_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    $defaults  = gudauri_map_default_settings();
    $settings  = gudauri_map_get_settings();
    $updated   = false;

    if (isset($_POST['gudauri_map_nonce']) && wp_verify_nonce($_POST['gudauri_map_nonce'], 'gudauri_map_save')) {
        $fields = array_keys($defaults);
        foreach ($fields as $field) {
            $raw = isset($_POST[$field]) ? wp_unslash($_POST[$field]) : '';
            if (strpos($field, 'map_url') === 0) {
                $settings[$field] = esc_url_raw($raw);
            } elseif (strpos($field, '_body') !== false) {
                $settings[$field] = wp_kses_post($raw);
            } else {
                $settings[$field] = sanitize_text_field($raw);
            }
        }
        update_option('gudauri_map_settings', $settings);
        $updated = true;
    }
    ?>
    <div class="wrap">
      <h1><?php esc_html_e('Gudauri Map', 'gudauri-map'); ?></h1>
      <?php if ($updated) : ?>
        <div class="updated notice"><p><?php esc_html_e('Settings saved.', 'gudauri-map'); ?></p></div>
      <?php endif; ?>
      <form method="post">
        <?php wp_nonce_field('gudauri_map_save', 'gudauri_map_nonce'); ?>
        <table class="form-table">
          <?php
          $groups = array(
              'tb'  => 'Tbilisi',
              'mt'  => 'Mtskheta',
              'zhi' => 'Zhinvali',
              'ana' => 'Ananuri',
              'pas' => 'Pasanauri',
          );
          $pins = array(
              'pin_mt' => 'Mtskheta',
              'pin_zhi' => 'Zhinvali',
              'pin_ana' => 'Ananuri',
              'pin_pas' => 'Pasanauri',
          );
          foreach ($groups as $prefix => $label) :
              $title_key = "{$prefix}_title";
              $meta_key  = "{$prefix}_meta";
              $body_key  = "{$prefix}_body";
              ?>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($title_key); ?>"><?php echo esc_html($label); ?> Title</label></th>
                <td><input type="text" class="regular-text" id="<?php echo esc_attr($title_key); ?>" name="<?php echo esc_attr($title_key); ?>" value="<?php echo esc_attr($settings[$title_key]); ?>"></td>
              </tr>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($meta_key); ?>"><?php echo esc_html($label); ?> Meta</label></th>
                <td><input type="text" class="regular-text" id="<?php echo esc_attr($meta_key); ?>" name="<?php echo esc_attr($meta_key); ?>" value="<?php echo esc_attr($settings[$meta_key]); ?>"></td>
              </tr>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($body_key); ?>"><?php echo esc_html($label); ?> Body</label></th>
                <td><textarea class="large-text" rows="4" id="<?php echo esc_attr($body_key); ?>" name="<?php echo esc_attr($body_key); ?>"><?php echo esc_textarea($settings[$body_key]); ?></textarea></td>
              </tr>
              <tr><td colspan="2"><hr></td></tr>
          <?php endforeach; ?>
        </table>
        <h2><?php esc_html_e('Pin Labels', 'gudauri-map'); ?></h2>
        <table class="form-table">
          <?php foreach ($pins as $key => $label) : ?>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($key); ?>"><?php echo esc_html($label); ?> Pin</label></th>
                <td><input type="text" class="regular-text" id="<?php echo esc_attr($key); ?>" name="<?php echo esc_attr($key); ?>" value="<?php echo esc_attr($settings[$key]); ?>"></td>
              </tr>
          <?php endforeach; ?>
        </table>
        <h2><?php esc_html_e('Map Image', 'gudauri-map'); ?></h2>
        <table class="form-table">
          <tr>
            <th scope="row"><label for="map_url"><?php esc_html_e('Map SVG URL', 'gudauri-map'); ?></label></th>
            <td>
              <input type="text" class="regular-text" id="map_url" name="map_url" value="<?php echo esc_attr($settings['map_url']); ?>">
              <button type="button" class="button gudauri-map-upload" data-target="map_url"><?php esc_html_e('Upload', 'gudauri-map'); ?></button>
            </td>
          </tr>
        </table>
        <h2><?php esc_html_e('Georgian (ka)', 'gudauri-map'); ?></h2>
        <table class="form-table">
          <?php foreach ($groups as $prefix => $label) :
              $title_key = "{$prefix}_title_ka";
              $meta_key  = "{$prefix}_meta_ka";
              $body_key  = "{$prefix}_body_ka";
              ?>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($title_key); ?>"><?php echo esc_html($label); ?> Title (ka)</label></th>
                <td><input type="text" class="regular-text" id="<?php echo esc_attr($title_key); ?>" name="<?php echo esc_attr($title_key); ?>" value="<?php echo esc_attr($settings[$title_key]); ?>"></td>
              </tr>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($meta_key); ?>"><?php echo esc_html($label); ?> Meta (ka)</label></th>
                <td><input type="text" class="regular-text" id="<?php echo esc_attr($meta_key); ?>" name="<?php echo esc_attr($meta_key); ?>" value="<?php echo esc_attr($settings[$meta_key]); ?>"></td>
              </tr>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($body_key); ?>"><?php echo esc_html($label); ?> Body (ka)</label></th>
                <td><textarea class="large-text" rows="4" id="<?php echo esc_attr($body_key); ?>" name="<?php echo esc_attr($body_key); ?>"><?php echo esc_textarea($settings[$body_key]); ?></textarea></td>
              </tr>
              <tr><td colspan="2"><hr></td></tr>
          <?php endforeach; ?>
        </table>
        <h3><?php esc_html_e('Pin Labels (ka)', 'gudauri-map'); ?></h3>
        <table class="form-table">
          <?php foreach ($pins as $key => $label) :
              $pin_key = $key . '_ka';
              ?>
              <tr>
                <th scope="row"><label for="<?php echo esc_attr($pin_key); ?>"><?php echo esc_html($label); ?> Pin (ka)</label></th>
                <td><input type="text" class="regular-text" id="<?php echo esc_attr($pin_key); ?>" name="<?php echo esc_attr($pin_key); ?>" value="<?php echo esc_attr($settings[$pin_key]); ?>"></td>
              </tr>
          <?php endforeach; ?>
        </table>
        <h3><?php esc_html_e('Map Image (ka)', 'gudauri-map'); ?></h3>
        <table class="form-table">
          <tr>
            <th scope="row"><label for="map_url_ka"><?php esc_html_e('Map SVG URL (ka)', 'gudauri-map'); ?></label></th>
            <td>
              <input type="text" class="regular-text" id="map_url_ka" name="map_url_ka" value="<?php echo esc_attr($settings['map_url_ka']); ?>">
              <button type="button" class="button gudauri-map-upload" data-target="map_url_ka"><?php esc_html_e('Upload', 'gudauri-map'); ?></button>
            </td>
          </tr>
        </table>
        <?php submit_button(); ?>
      </form>
      <p><?php esc_html_e('Use the shortcode [gudauri_map] in any page or post.', 'gudauri-map'); ?></p>
    </div>
    <?php
}

/**
 * Ensure defaults exist on activation.
 */
function gudauri_map_activate() {
    if (!get_option('gudauri_map_settings')) {
        add_option('gudauri_map_settings', gudauri_map_default_settings());
    }
}
register_activation_hook(__FILE__, 'gudauri_map_activate');
